define([
    'backbone',
    'Layout',
    'modules/new-appointment-request/views/direct/preferred-date/closest-dates',
    'modules/new-appointment-request/views/direct/preferred-date/date-collapsible',
    'modules/new-appointment-request/resources/direct/preferred-date/time-slots-by-date-collection',
], function(Backbone, Layout, ClosestDatesView, DateCollapsibleView, TimeSlotsByDateCollection) {
    'use strict';

    describe('Closest Dates View', function() {
        var view;
        var $view;
        var model;
        var timeSlotsByDateCollection;

        beforeEach(function() {
            spyOn(ClosestDatesView.prototype, '_initializeAndFilterCollection').and.callThrough();
            timeSlotsByDateCollection = new TimeSlotsByDateCollection([
                {date: '10/08/2015'},
                {date: '10/09/2015'},
                {date: '10/11/2015'},
                {date: '10/13/2015'},
                {date: '10/15/2015'},
            ]);
        });
        afterEach(function() {
            view.destroy();
        });

        describe('on show', function() {
            var layout = new Layout();

            layout.render();

            beforeEach(function() {
                spyOn(ClosestDatesView.prototype, 'focusOnLegend').and.callThrough();

                model = new Backbone.Model({desiredDate: '10/12/2015 00:00:00'});

                view = new ClosestDatesView({
                    model: model,
                    timeSlotsByDateCollection: timeSlotsByDateCollection,
                });

                layout.showChildView('content', view);
                $view = view.$el;
            });

            it('has a legend and a description', function() {
                expect($view.find('legend').text())
                    .toEqual('*\xa0Select Date/Time:');
                expect($view.find('#available-dates-times-description').text())
                    .toEqual('No appointments are available on your preferred date. These are the closest available dates. Select "Show All Dates" to see all date options.');
            });
            it('has a childViewContainer', function () {
                expect($view.has(view.childViewContainer).length).toEqual(1);
            });
            it('has DateCollapsibleView as its childView', function () {
                expect(view.childView).toBe(DateCollapsibleView);
            });
            xit('calls function to focusOnLegend()', function() {
                // unable to test focus in headless browser
                expect(view.focusOnLegend).toHaveBeenCalled();
                expect($.find('legend span[tabindex]').length).toEqual(1);
            });
        });
        describe('when initializing this.collection to the three dates closest to the desiredDate', function() {
            it('returns the earlier date when an earlier date and a later date tie in difference', function() {
                model = new Backbone.Model({desiredDate: '10/12/2015 00:00:00'});

                view = new ClosestDatesView({
                    model: model,
                    timeSlotsByDateCollection: timeSlotsByDateCollection,
                });
                expect(view._initializeAndFilterCollection).toHaveBeenCalled();
                expect(view.collection.length).toEqual(3);
                expect(view.collection.at(0).get('date')).toEqual('10/09/2015');
                expect(view.collection.at(1).get('date')).toEqual('10/11/2015');
                expect(view.collection.at(2).get('date')).toEqual('10/13/2015');
            });
            it('returns all dates when the array contains three or fewer dates total', function() {
                model = new Backbone.Model({desiredDate: '10/12/2015 00:00:00'});

                view = new ClosestDatesView({
                    model: model,
                    timeSlotsByDateCollection: new TimeSlotsByDateCollection([
                        {date: '10/09/2015'},
                        {date: '10/15/2015'},
                    ]),
                });
                expect(view._initializeAndFilterCollection).toHaveBeenCalled();
                expect(view.collection.length).toEqual(2);
                expect(view.collection.at(0).get('date')).toEqual('10/09/2015');
                expect(view.collection.at(1).get('date')).toEqual('10/15/2015');
            });
            it('returns the earliest three dates when the desiredDate is before the start of the array', function() {
                model = new Backbone.Model({desiredDate: '10/01/2015 00:00:00'});

                view = new ClosestDatesView({
                    model: model,
                    timeSlotsByDateCollection: timeSlotsByDateCollection,
                });
                expect(view._initializeAndFilterCollection).toHaveBeenCalled();
                expect(view.collection.length).toEqual(3);
                expect(view.collection.at(0).get('date')).toEqual('10/08/2015');
                expect(view.collection.at(1).get('date')).toEqual('10/09/2015');
                expect(view.collection.at(2).get('date')).toEqual('10/11/2015');
            });
            it('returns the latest three dates when the desiredDate is after the end of the array', function() {
                model = new Backbone.Model({desiredDate: '10/26/2015 00:00:00'});

                view = new ClosestDatesView({
                    model: model,
                    timeSlotsByDateCollection: timeSlotsByDateCollection,
                });
                expect(view._initializeAndFilterCollection).toHaveBeenCalled();
                expect(view.collection.length).toEqual(3);
                expect(view.collection.at(0).get('date')).toEqual('10/11/2015');
                expect(view.collection.at(1).get('date')).toEqual('10/13/2015');
                expect(view.collection.at(2).get('date')).toEqual('10/15/2015');
            });
        });
    });

}

);
